<?php
/*------------------------------------------------------------------------------
 StatisticsServiceProvider.php 2021-08-26
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2021 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 -----------------------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Language\Services\LanguageFactory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Data\PreferredPeriodStorage as DashboardWidgetPreferredPeriodStorage;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Data\PreferredCategoryStorage as DashboardWidgetPreferredCategoryStorage;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Service as DashboardWidgetService;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Factories\ConversionDataProviderResultFactory as DashboardWidgetConversionDataProviderResultFactory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Factories\OrdersDataProviderResultFactory as DashboardWidgetOrdersDataProviderResultFactory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Factories\SalesDataProviderResultFactory as DashboardWidgetSalesDataProviderResultFactory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Factories\SummarizableTimespanFactory as DashboardWidgetSummarizableTimespanFactory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Factories\VisitorsDataProviderResultFactory as DashboardWidgetVisitorsDataProviderResultFactory;
use Gambio\Admin\Modules\Statistics\App\DashboardWidget\Services\Mapper\CompatibilityMapper as DashboardWidgetCompatibilityMapper;
use Gambio\Admin\Modules\Statistics\App\Data\Data\Mapper as DataMapper;
use Gambio\Admin\Modules\Statistics\App\Data\Data\Reader as DataReader;
use Gambio\Admin\Modules\Statistics\App\Data\Data\Repository as DataRepository;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Data\Factory\Map as MapDataFactory;
use Gambio\Admin\Modules\Statistics\App\Data\Factory\Number as NumberDataFactory;
use Gambio\Admin\Modules\Statistics\App\Data\Factory\Serial as SerialDataFactory;
use Gambio\Admin\Modules\Statistics\App\Data\Factory\Table as TableDataFactory;
use Gambio\Admin\Modules\Statistics\App\Data\Factory\Text as TextDataFactory;
use Gambio\Admin\Modules\Statistics\App\Data\Factory\Timespan as TimespanFactory;
use Gambio\Admin\Modules\Statistics\App\Data\Service as DataService;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\AddWidget as OverviewAddWidgetAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\ConfigureWidget as OverviewConfigureWidgetAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\DeleteWidget as OverviewDeleteWidgetAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\GetAvailableWidgets as OverviewGetAvailableWidgetsAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\GetWidget as OverviewGetWidgetAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\GetWidgets as OverviewGetWidgetsAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\MoveWidget as OverviewMoveWidgetAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\Overview as OverviewAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\RenameWidget as OverviewRenameWidgetAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Action\ResizeWidget as OverviewResizeWidgetAction;
use Gambio\Admin\Modules\Statistics\App\Overview\Data\Collector as OverviewCollector;
use Gambio\Admin\Modules\Statistics\App\Overview\Data\Database as OverviewDatabase;
use Gambio\Admin\Modules\Statistics\App\Overview\Data\Mapper as OverviewMapper;
use Gambio\Admin\Modules\Statistics\App\Overview\Data\Reader as OverviewReader;
use Gambio\Admin\Modules\Statistics\App\Overview\Data\Repository as OverviewRepository;
use Gambio\Admin\Modules\Statistics\App\Overview\Data\Writer as OverviewWriter;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Connection as ConnectionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option as OverviewOptionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Dropdown as OverviewDropdownOptionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined as OverviewPredefinedOptionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown as OverviewTimespanDropdownOptionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Visualization as OverviewVisualizationFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Schemes as OverviewSchemes;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioCategoriesRankingOverview as GambioCategoriesRankingOverviewWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioConversionRateCount as GambioConversionRateCountDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioConversionRateTrend as GambioConversionRateTrendWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioCustomersRanking as GambioCustomersRankingWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioCustomersSignupsCount as GambioCustomersSignupsCountWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioCustomersSignupsTrend as GambioCustomersSignupsTrendWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioOrdersCount as GambioOrdersCountWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioOrdersList as GambioOrdersListWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioOrdersTrend as GambioOrdersTrendWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioProductsRankingOverview as GambioProductsRankingOverviewWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioRevenueCount as GambioRevenueCountWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioRevenueTrend as GambioRevenueTrendWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioVisitorsCount as GambioVisitorsCountWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioVisitorsOnlineCount as GambioVisitorsOnlineCountWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioVisitorsTrend as GambioVisitorsTrendWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioWithdrawalsTrend as GambioWithdrawalsTrendWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition\GambioWithdrawalsCount as GambioWithdrawalsCountWidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Schema as OverviewSchema;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Schema\Provider as OverviewSchemaProvider;
use Gambio\Admin\Modules\Statistics\App\Overview\SchemaValidator as OverviewSchemaValidator;
use Gambio\Admin\Modules\Statistics\App\Overview\Service as OverviewService;
use Gambio\Admin\Modules\Statistics\Services\StatisticsDashboardWidgetService as DashboardWidgetServiceInterface;
use Gambio\Admin\Modules\Statistics\Services\StatisticsDataService as DataServiceInterface;
use Gambio\Admin\Modules\Statistics\Services\StatisticsOverviewService as OverviewServiceInterface;
use Gambio\Core\Application\DependencyInjection\AbstractBootableServiceProvider;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Configuration\Compatibility\ConfigurationStorageRepositoryBuilder;
use Gambio\Core\Configuration\Services\ConfigurationService;
use Gambio\Core\Language\Services\LanguageService;
use Gambio\Core\TextManager\Services\TextManager;
use Gambio\Core\UserConfiguration\Services\CurrentUserConfigurationService;
use NumberFormatter;
use Psr\EventDispatcher\EventDispatcherInterface;
use Swaggest\JsonSchema\Context as JsonSchemaContext;

/**
 * @codeCoverageIgnore
 */
class StatisticsServiceProvider extends AbstractBootableServiceProvider
{
    private const OVERVIEW_SCHEMES = [
        "https://gambio.com/schema/AddWidgetRequestBody.json"       => __DIR__
                                                                       . "/App/Overview/Model/ValueObjects/Schema/Json/AddWidgetRequestBody.json",
        "https://gambio.com/schema/ConfigureWidgetRequestBody.json" => __DIR__
                                                                       . "/App/Overview/Model/ValueObjects/Schema/Json/ConfigureWidgetRequestBody.json",
        "https://gambio.com/schema/DatabaseEntry.json"              => __DIR__
                                                                       . "/App/Overview/Model/ValueObjects/Schema/Json/DatabaseEntry.json",
        "https://gambio.com/schema/MoveWidgetRequestBody.json"      => __DIR__
                                                                       . "/App/Overview/Model/ValueObjects/Schema/Json/MoveWidgetRequestBody.json",
        "https://gambio.com/schema/ResizeWidgetRequestBody.json"    => __DIR__
                                                                       . "/App/Overview/Model/ValueObjects/Schema/Json/ResizeWidgetRequestBody.json",
        "https://gambio.com/schema/RenameWidgetRequestBody.json"    => __DIR__
                                                                       . "/App/Overview/Model/ValueObjects/Schema/Json/RenameWidgetRequestBody.json",
    ];
    
    /**
     * @var string
     */
    private $languageCode;
    
    /**
     * @var NumberFormatter
     */
    private $currencyNumberFormatter;
    
    /**
     * @var NumberFormatter
     */
    private $localeNumberFormatter;
    
    /**
     * @var NumberFormatter
     */
    private $percentageNumberFormatter;
    
    
    /**
     * @inheritDoc
     */
    public function provides(): array
    {
        return [
            DashboardWidgetServiceInterface::class,
            DataFactory::class,
            DataServiceInterface::class,
            OverviewServiceInterface::class,
            OverviewAddWidgetAction::class,
            OverviewConfigureWidgetAction::class,
            OverviewDeleteWidgetAction::class,
            OverviewGetAvailableWidgetsAction::class,
            OverviewGetWidgetsAction::class,
            OverviewMoveWidgetAction::class,
            OverviewAction::class,
            OverviewResizeWidgetAction::class,
            OverviewRenameWidgetAction::class,
            OverviewGetWidgetAction::class,
        ];
    }
    
    
    public function boot(): void
    {
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioCategoriesRankingOverviewWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioConversionRateTrendWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioConversionRateCountDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioCustomersRankingWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioCustomersSignupsCountWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioCustomersSignupsTrendWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioCustomersSignupsCountWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioOrdersCountWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioOrdersListWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioOrdersTrendWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioProductsRankingOverviewWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioRevenueCountWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioRevenueTrendWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioVisitorsCountWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioVisitorsTrendWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioVisitorsOnlineCountWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioWithdrawalsTrendWidgetDefinition::class]);
        $this->application->inflect(OverviewCollector::class)
            ->invokeMethod('registerWidgetDefinition', [GambioWithdrawalsCountWidgetDefinition::class]);
    }
    
    
    /**
     * @inheritDoc
     */
    public function register(): void
    {
        $this->registerSystemDependencies();
        $this->registerDashboardWidget();
        $this->registerData();
        $this->registerOverviewActions();
        $this->registerOverviewSchemes();
        $this->registerOverviewServices();
        $this->registerOverviewFactories();
        $this->registerOverviewData();
        $this->registerWidgetDefinitions();
    }
    
    
    /**
     * Register system dependencies.
     */
    private function registerSystemDependencies(): void
    {
        $this->languageCode              = $this->application->get(UserPreferences::class)->languageId()
                                           === 2 ? 'de' : 'en';
        $this->currencyNumberFormatter   = new NumberFormatter($this->languageCode, NumberFormatter::CURRENCY);
        $this->localeNumberFormatter     = new NumberFormatter($this->languageCode, NumberFormatter::DEFAULT_STYLE);
        $this->percentageNumberFormatter = new NumberFormatter($this->languageCode, NumberFormatter::PERCENT);
    }
    
    
    /**
     * Register dependencies for dashboard statistics widget.
     */
    private function registerDashboardWidget(): void
    {
        $this->application->registerShared(DashboardWidgetCompatibilityMapper::class)
            ->addArgument(DashboardWidgetOrdersDataProviderResultFactory::class)
            ->addArgument(DashboardWidgetConversionDataProviderResultFactory::class)
            ->addArgument(DashboardWidgetSalesDataProviderResultFactory::class)
            ->addArgument(DashboardWidgetVisitorsDataProviderResultFactory::class);
        
        $this->application->registerShared(DashboardWidgetSummarizableTimespanFactory::class)
            ->addArgument(TextManager::class);
        
        $this->application->registerShared(DashboardWidgetPreferredPeriodStorage::class)
            ->addArgument(ConfigurationStorageRepositoryBuilder::class);
        
        $this->application->registerShared(DashboardWidgetPreferredCategoryStorage::class)
            ->addArgument(ConfigurationStorageRepositoryBuilder::class);
        
        $this->application->registerShared(DashboardWidgetConversionDataProviderResultFactory::class)
            ->addArgument(TextManager::class);
        
        $this->application->registerShared(DashboardWidgetOrdersDataProviderResultFactory::class)
            ->addArgument(TextManager::class);
        
        $this->application->registerShared(DashboardWidgetVisitorsDataProviderResultFactory::class)
            ->addArgument(TextManager::class);
        
        $this->application->registerShared(DashboardWidgetSalesDataProviderResultFactory::class)
            ->addArgument(TextManager::class)
            ->addArgument(ConfigurationService::class);
        
        $this->application->registerShared(DashboardWidgetServiceInterface::class, DashboardWidgetService::class)
            ->addArgument(DashboardWidgetSummarizableTimespanFactory::class)
            ->addArgument(DashboardWidgetPreferredPeriodStorage::class)
            ->addArgument(DashboardWidgetPreferredCategoryStorage::class)
            ->addArgument(DataServiceInterface::class)
            ->addArgument(DashboardWidgetCompatibilityMapper::class);
    }
    
    
    /**
     * Register data classes.
     */
    private function registerData(): void
    {
        $this->application->registerShared(DataMapper::class)->addArgument(DataFactory::class);
        
        $this->application->registerShared(DataReader::class)->addArgument(Connection::class);
        
        $this->application->registerShared(DataRepository::class)
            ->addArgument(DataReader::class)
            ->addArgument(DataMapper::class);
        
        $this->application->registerShared(MapDataFactory::class);
        
        $this->application->registerShared(SerialDataFactory::class);
        
        $this->application->registerShared(TableDataFactory::class);
        
        $this->application->registerShared(TextDataFactory::class);
        
        $this->application->registerShared(NumberDataFactory::class);
        
        $this->application->registerShared(DataFactory::class)
            ->addArgument(MapDataFactory::class)
            ->addArgument(SerialDataFactory::class)
            ->addArgument(TableDataFactory::class)
            ->addArgument(TextDataFactory::class)
            ->addArgument(NumberDataFactory::class)
            ->addArgument(TimespanFactory::class);
        
        $this->application->register(TimespanFactory::class);
        
        $this->application->registerShared(DataServiceInterface::class, DataService::class)
            ->addArgument(DataRepository::class)
            ->addArgument(DataFactory::class);
    }
    
    
    /**
     * Register overview action classes.
     */
    private function registerOverviewActions(): void
    {
        $this->application->registerShared(OverviewAddWidgetAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(OverviewSchemaValidator::class);
        
        $this->application->registerShared(OverviewConfigureWidgetAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(OverviewSchemaValidator::class);
        
        $this->application->registerShared(OverviewDeleteWidgetAction::class)
            ->addArgument(OverviewServiceInterface::class);
        
        $this->application->registerShared(OverviewGetAvailableWidgetsAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(LanguageService::class)
            ->addArgument(UserPreferences::class);
        
        $this->application->registerShared(OverviewGetWidgetsAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(LanguageService::class)
            ->addArgument(UserPreferences::class);
        
        $this->application->registerShared(OverviewGetWidgetAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(LanguageService::class)
            ->addArgument(UserPreferences::class);
        
        $this->application->registerShared(OverviewMoveWidgetAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(OverviewSchemaValidator::class);
        
        $this->application->registerShared(OverviewAction::class);
        
        $this->application->registerShared(OverviewResizeWidgetAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(OverviewSchemaValidator::class);
        
        $this->application->registerShared(OverviewRenameWidgetAction::class)
            ->addArgument(OverviewServiceInterface::class)
            ->addArgument(OverviewSchemaValidator::class);
    }
    
    
    /**
     * Register overview data classes.
     */
    private function registerOverviewData(): void
    {
        $this->application->registerShared(OverviewCollector::class)->addArgument(OverviewFactory::class);
        
        $this->application->registerShared(OverviewDatabase::class)->addArgument(ConnectionFactory::class);
        
        $this->application->registerShared(OverviewMapper::class)->addArgument(OverviewFactory::class);
        
        $this->application->registerShared(OverviewReader::class)
            ->addArgument(OverviewDatabase::class)
            ->addArgument(OverviewCollector::class);
        
        $this->application->registerShared(OverviewRepository::class)
            ->addArgument(OverviewReader::class)
            ->addArgument(OverviewWriter::class)
            ->addArgument(OverviewMapper::class)
            ->addArgument(EventDispatcherInterface::class);
        
        $this->application->registerShared(OverviewWriter::class)
            ->addArgument(OverviewDatabase::class)
            ->addArgument(OverviewSchemaValidator::class);
    }
    
    
    /**
     * Register overview factory classes.
     */
    private function registerOverviewFactories(): void
    {
        $this->application->registerShared(ConnectionFactory::class)
            ->addArgument(CurrentUserConfigurationService::class);
        
        $this->application->registerShared(OverviewTimespanDropdownOptionFactory::class);
        
        $this->application->registerShared(OverviewDropdownOptionFactory::class);
        
        $this->application->registerShared(OverviewPredefinedOptionFactory::class);
        
        $this->application->registerShared(OverviewOptionFactory::class)
            ->addArgument(OverviewDropdownOptionFactory::class)
            ->addArgument(OverviewPredefinedOptionFactory::class);
        
        $this->application->registerShared(OverviewVisualizationFactory::class);
        
        $this->application->registerShared(OverviewFactory::class)
            ->addArgument(OverviewOptionFactory::class)
            ->addArgument(OverviewVisualizationFactory::class)
            ->addArgument(LanguageFactory::class);
    }
    
    
    /**
     * Register overview service classes.
     */
    private function registerOverviewServices(): void
    {
        $this->application->registerShared(OverviewSchemaValidator::class)->addArgument(OverviewSchemes::class);
        
        $this->application->registerShared(OverviewServiceInterface::class, OverviewService::class)
            ->addArgument(OverviewRepository::class)
            ->addArgument(OverviewFactory::class);
    }
    
    
    /**
     * Register overview schemes.
     */
    private function registerOverviewSchemes(): void
    {
        $this->application->registerShared(OverviewSchemaProvider::class);
        
        $context = new JsonSchemaContext($this->application->get(OverviewSchemaProvider::class));
        
        $this->application->registerShared(OverviewSchemes::class,
            function () use ($context) {
                return OverviewSchemes::create(...array_map(function (string $id, string $schema) use ($context) {
                    return OverviewSchema::create($id, $schema, $context);
                },
                    array_keys(self::OVERVIEW_SCHEMES),
                    self::OVERVIEW_SCHEMES));
            });
    }
    
    
    /**
     * Register widget definitions.
     */
    private function registerWidgetDefinitions(): void
    {
        $this->application->registerShared(GambioCategoriesRankingOverviewWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioConversionRateTrendWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioConversionRateCountDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class)
            ->addArgument($this->percentageNumberFormatter);
        
        $this->application->registerShared(GambioCustomersRankingWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class);
    
        $this->application->registerShared(GambioCustomersSignupsCountWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class)
            ->addArgument($this->localeNumberFormatter);
        
        $this->application->registerShared(GambioCustomersSignupsTrendWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioOrdersCountWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class)
            ->addArgument($this->localeNumberFormatter);
        
        $this->application->registerShared(GambioOrdersListWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class)
            ->addArgument($this->currencyNumberFormatter);
        
        $this->application->registerShared(GambioOrdersTrendWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioProductsRankingOverviewWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioRevenueCountWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class)
            ->addArgument($this->currencyNumberFormatter);
        
        $this->application->registerShared(GambioRevenueTrendWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioVisitorsCountWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class)
            ->addArgument($this->localeNumberFormatter);
        
        $this->application->registerShared(GambioVisitorsOnlineCountWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class)
            ->addArgument($this->localeNumberFormatter);
        
        $this->application->registerShared(GambioVisitorsTrendWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioWithdrawalsTrendWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(UserPreferences::class)
            ->addArgument(Connection::class);
        
        $this->application->registerShared(GambioWithdrawalsCountWidgetDefinition::class)
            ->addArgument(OverviewFactory::class)
            ->addArgument(DataFactory::class)
            ->addArgument(Connection::class)
            ->addArgument($this->localeNumberFormatter);
    }
}